/*******************************************************************************
 * ALBERTA:  an Adaptive multi Level finite element toolbox using
 *           Bisectioning refinement and Error control by Residual
 *           Techniques for scientific Applications
 *
 * file:     disc-ortho-poly.c
 *
 * description: Orthogonal discontinuous polynomials of degree 1 and 2.
 *
 *******************************************************************************
 *
 *  authors:   Claus-Justus Heine
 *             Abteilung fuer Angewandte Mathematik
 *             Albert-Ludwigs-Universitaet Freiburg
 *             Hermann-Herder-Str. 10
 *             D-79104 Freiburg im Breisgau, Germany
 *
 *  http://www.alberta-fem.de
 *
 *  (c) by C.-J. Heine (2008)
 *
 ******************************************************************************/

#include "alberta.h"

#define COMPILED_CONSTANTS 0

#if COMPILED_CONSTANTS
# define C         (0.25*(1.0 - sqrt(5.0)/5.0))
# define WEIGHT_V  (2.0*sqrt(30.0))
# define WEIGHT_E					\
  (210.0/sqrt((9100.0+2870.0*sqrt(10.0)			\
	       -					\
	       2345*sqrt(15.0)				\
	       -					\
	       3710*sqrt(6.0))))
#else
# define C 0.1381966011250105151795413165634361882279690820194237137864551378
# define WEIGHT_V							\
  10.95445115010332226913939565601604267905489389995966508453788899
# define WEIGHT_E							\
  85.84673630483648681934404929933852987186250258634791528912539820
#endif

static ORTHO_DATA d_ortho_1_3d_data;
static ORTHO_DATA d_ortho_2_3d_data;

/* p.w. linear orthogonal polynomials */
static inline REAL d_ortho_phi_v_3d(const REAL_B lambda, int v)
{
  return WEIGHT_V * (lambda[v] - C);
}

static inline const REAL *
d_ortho_grd_phi_v_3d(REAL_B result, const REAL_B lambda, int v, bool doinit)
{
  if (doinit) {
    SET_BAR(3 /* dim */, 0.0, result);
  }

  result[v] = WEIGHT_V;

  return result;
}

static inline const REAL_B *d_ortho_D2_phi_v_3d(REAL_BB result,
						const REAL_B lambda, int v,
						bool doinit)
{
  if (doinit) {
    MSET_BAR(3 /* dim */, 0.0, result);
  }
  
  return (const REAL_B *)result;
}

/* values */
static REAL d_ortho_phi_v0_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_v_3d(lambda, 0);
}

static REAL d_ortho_phi_v1_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_v_3d(lambda, 1);
}

static REAL d_ortho_phi_v2_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_v_3d(lambda, 2);
}

static REAL d_ortho_phi_v3_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_v_3d(lambda, 3);
}

/* gradients */
static const REAL *
d_ortho_grd_phi_v0_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_v_3d(grd, lambda, 0, false);
}

static const REAL *
d_ortho_grd_phi_v1_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_v_3d(grd, lambda, 1, false);
}

static const REAL *
d_ortho_grd_phi_v2_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_v_3d(grd, lambda, 2, false);
}

static const REAL *
d_ortho_grd_phi_v3_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_v_3d(grd, lambda, 3, false);
}

/* Hessians */
static const REAL_B *
d_ortho_D2_phi_v0_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_v_3d(D2, lambda, 0, false);
}

static const REAL_B *
d_ortho_D2_phi_v1_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_v_3d(D2, lambda, 1, false);
}

static const REAL_B *
d_ortho_D2_phi_v2_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;
  
  return d_ortho_D2_phi_v_3d(D2, lambda, 2, false);
}

static const REAL_B *
d_ortho_D2_phi_v3_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;
  
  return d_ortho_D2_phi_v_3d(D2, lambda, 2, false);
}

/*******************************************************************************
 *
 * p.w. quadratic orthogonal add-on
 *
 ******************************************************************************/
#if COMPILED_CONSTANTS
# define K \
  (-10.0+3.0*sqrt(15)+3.0*sqrt(6.0)-3.0*sqrt(10.0)			\
   +									\
   2.0*sqrt(10.0)*(-1.5+0.5*sqrt(15.0)+0.5*sqrt(6.0)-0.5*sqrt(10.0)))
# define L (-1.5+0.5*sqrt(15)+0.5*sqrt(6)-0.5*sqrt(10.0))
# define E								\
  (5.0/48.0*sqrt(6.0)-7.0/80.0*sqrt(10.0)-0.25+sqrt(15.0)/15.0		\
   +									\
   7.0/240.0*sqrt(30.0)-5.0/48..0*sqrt(2.0)-sqrt(5.0)/15.0+sqrt(3.0)/12.0)
# define F								\
  (-7.0/240.0*sqrt(30.0)+5.0/48.0*sqrt(6.0)-7.0/80.0*sqrt(10.0)-0.25	\
   +									\
   sqrt(15.0)/15.0+5.0/48.0*sqrt(2.0)+sqrt(5.0)/15.0-sqrt(3.0)/12.0)
#else
# define K -1.28312877077667805187616700371820124251424642399858007456197434e-2
# define L  8.0097714411107825688827965027786234539657023311522064081380741e-2
# define E -5.640355321761170434285838948711190214077835570370916618923774182e-3
# define F -2.105009263355419085051762912563389841621569463585697444598545489e-2

#endif

static inline REAL d_ortho_phi_e_3d(const REAL_B lambda, int e)
{
  const int (*voe)[N_VERTICES_1D+1] = vertex_of_edge_3d;
  int i0 = voe[e][0], i1 = voe[e][1];
  int i2 = voe[N_EDGES_3D-1-e][0], i3 = voe[N_EDGES_3D-1-e][1];

  return
    WEIGHT_E
    *
    (lambda[i0]*lambda[i1]
     +
     K * lambda[i2]*lambda[i3]
     +
     L * (lambda[i0]*lambda[i2] + lambda[i0]*lambda[i3]
	  +
	  lambda[i1]*lambda[i2] + lambda[i1]*lambda[i3])
     +
     F * (d_ortho_phi_v_3d(lambda, i0) + d_ortho_phi_v_3d(lambda, i1))
     +
     E * (d_ortho_phi_v_3d(lambda, i2) + d_ortho_phi_v_3d(lambda, i3)));
}
    
static inline const REAL *
d_ortho_grd_phi_e_3d(REAL_B grd, const REAL_B lambda, int e)
{
  REAL_B tmp;
  const int (*voe)[N_VERTICES_1D+1] = vertex_of_edge_3d;
  int i0 = voe[e][0], i1 = voe[e][1];
  int i2 = voe[N_EDGES_3D-1-e][0], i3 = voe[N_EDGES_3D-1-e][1];

  grd[i0] = lambda[i1] + L * (lambda[i2] + lambda[i3]);
  grd[i1] = lambda[i0] + L * (lambda[i2] + lambda[i3]);
  grd[i2] = K * lambda[i3] + L * (lambda[i0] + lambda[i1]);
  grd[i3] = K * lambda[i2] + L * (lambda[i0] + lambda[i1]);
  
  AXPY_BAR(3 /* dim */, F, d_ortho_grd_phi_v_3d(tmp, lambda, i0, true), grd);
  AXPY_BAR(3 /* dim */, F, d_ortho_grd_phi_v_3d(tmp, lambda, i1, true), grd);
  AXPY_BAR(3 /* dim */, E, d_ortho_grd_phi_v_3d(tmp, lambda, i2, true), grd);
  AXPY_BAR(3 /* dim */, E, d_ortho_grd_phi_v_3d(tmp, lambda, i3, true), grd);

  SCAL_BAR(3 /* dim */, WEIGHT_E, grd);
  
  return grd;
}

static inline const REAL_B *
d_ortho_D2_phi_e_3d(REAL_BB D2, const REAL_B lambda, int e)
{
  REAL_BB tmp = { { 0.0, } };
  const int (*voe)[N_VERTICES_1D+1] = vertex_of_edge_3d;
  int i0 = voe[e][0], i1 = voe[e][1];
  int i2 = voe[N_EDGES_3D-1-e][0], i3 = voe[N_EDGES_3D-1-e][1];

  D2[i0][i1] = D2[i1][i0] = 1.0;
  D2[i0][i2] = D2[i2][i0] = L;
  D2[i0][i3] = D2[i3][i0] = L;

  D2[i1][i2] = D2[i2][i1] = L;
  D2[i1][i3] = D2[i3][i1] = L;

  D2[i2][i3] = D2[i3][i2] = K;

  MAXPY_BAR(3 /* dim */, F, d_ortho_D2_phi_v_3d(tmp, lambda, i0, true), D2);
  MAXPY_BAR(3 /* dim */, F, d_ortho_D2_phi_v_3d(tmp, lambda, i1, true), D2);
  MAXPY_BAR(3 /* dim */, E, d_ortho_D2_phi_v_3d(tmp, lambda, i2, true), D2);
  MAXPY_BAR(3 /* dim */, E, d_ortho_D2_phi_v_3d(tmp, lambda, i3, true), D2);

  MSCAL_BAR(3 /* dim */, WEIGHT_E, D2);
  
  return (const REAL_B *)D2;
}

#undef C
#undef K
#undef L
#undef F
#undef E
#undef WEIGHT_E
#undef WEIGHT_V

/* values */
static REAL d_ortho_phi_e0_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_e_3d(lambda, 0);
}

static REAL d_ortho_phi_e1_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_e_3d(lambda, 1);
}

static REAL d_ortho_phi_e2_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_e_3d(lambda, 2);
}

static REAL d_ortho_phi_e3_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_e_3d(lambda, 3);
}

static REAL d_ortho_phi_e4_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_e_3d(lambda, 4);
}

static REAL d_ortho_phi_e5_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  return d_ortho_phi_e_3d(lambda, 5);
}

/* gradients */
static const REAL *
d_ortho_grd_phi_e0_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_e_3d(grd, lambda, 0);
}

static const REAL *
d_ortho_grd_phi_e1_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_e_3d(grd, lambda, 1);
}

static const REAL *
d_ortho_grd_phi_e2_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_e_3d(grd, lambda, 2);
}

static const REAL *
d_ortho_grd_phi_e3_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_e_3d(grd, lambda, 3);
}

static const REAL *
d_ortho_grd_phi_e4_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_e_3d(grd, lambda, 4);
}

static const REAL *
d_ortho_grd_phi_e5_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_B grd;

  return d_ortho_grd_phi_e_3d(grd, lambda, 5);
}

/* Hessians */
static const REAL_B *
d_ortho_D2_phi_e0_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_e_3d(D2, lambda, 0);
}

static const REAL_B *
d_ortho_D2_phi_e1_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_e_3d(D2, lambda, 1);
}

static const REAL_B *
d_ortho_D2_phi_e2_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_e_3d(D2, lambda, 2);
}


static const REAL_B *
d_ortho_D2_phi_e3_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_e_3d(D2, lambda, 3);
}


static const REAL_B *
d_ortho_D2_phi_e4_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_e_3d(D2, lambda, 4);
}


static const REAL_B *
d_ortho_D2_phi_e5_3d(const REAL_B lambda, const BAS_FCTS *thisptr)
{
  static REAL_BB D2;

  return d_ortho_D2_phi_e_3d(D2, lambda, 5);
}

/* jump tables */

static const BAS_FCT d_ortho_phi_3d[N_BAS_LAG_2_3D] = {
  d_ortho_phi_v0_3d, d_ortho_phi_v1_3d, d_ortho_phi_v2_3d, d_ortho_phi_v3_3d,
  d_ortho_phi_e0_3d, d_ortho_phi_e1_3d, d_ortho_phi_e2_3d,
  d_ortho_phi_e3_3d, d_ortho_phi_e4_3d, d_ortho_phi_e5_3d
};

static const GRD_BAS_FCT d_ortho_grd_phi_3d[N_BAS_LAG_2_3D] = {
  d_ortho_grd_phi_v0_3d, d_ortho_grd_phi_v1_3d,
  d_ortho_grd_phi_v2_3d, d_ortho_grd_phi_v3_3d,
  d_ortho_grd_phi_e0_3d, d_ortho_grd_phi_e1_3d, d_ortho_grd_phi_e2_3d,
  d_ortho_grd_phi_e3_3d, d_ortho_grd_phi_e4_3d, d_ortho_grd_phi_e5_3d
};

static const D2_BAS_FCT d_ortho_D2_phi_3d[N_BAS_LAG_2_3D] = {
  d_ortho_D2_phi_v0_3d, d_ortho_D2_phi_v1_3d,
  d_ortho_D2_phi_v2_3d, d_ortho_D2_phi_v3_3d,
  d_ortho_D2_phi_e0_3d, d_ortho_D2_phi_e1_3d, d_ortho_D2_phi_e2_3d,
  d_ortho_D2_phi_e3_3d, d_ortho_D2_phi_e4_3d, d_ortho_D2_phi_e5_3d,
};

/* Interpolation during refinement */

/* <<< interpolation matrix for child 0 */

/* Interpolation to child0, we have
 *
 * lp == bar. coords on parent, lc on child0
 * 
 * lp = lc_0 e_0 + lc_2 e_3 + lc_1 e_2 + lc_3 1/2 (e_0 + e_1)
 *
 * lp_0 = lc_0 + 1/2 lc_3,
 * lp_1 = 1/2 lc_3
 * lp_2 = lc_1
 * lp_3 = lc_2
 *
 * The table was constructed by maple (see orthpoly-3d.mws)
 */
static const REAL child0refine_inter[10][10] = {
  { 1.154508497187473712051146708591409529430077294951440715533862155,
    -.1545084971874737120511467085914095294300772949514407155338621556,
    0,
    0,
    -.6530582494771754744398266018812554046880246378515774673927733836,
    -.1437169328132325190444687037972815542382228273965639708457830335,
    -.1437169328132325190444687037972815542382228273965639708457830335,
    .3909998649235090370405005811858557276517203971888379362826572804e-1,
    .3909998649235090370405005811858557276517203971888379362826572804e-1,
    .8379578288972173185577336528559701739540157919035077699310042797e-2,
  },
  { .1545084971874737120511467085914095294300772949514407155338621556,
    -.1545084971874737120511467085914095294300772949514407155338621556,
    1,
    0,
    0,
    .5093413166639429553953578980839738504498018104550134965469903497,
    -.3072040820337873051847272159002587102563188179984871592895569129e-1,
    -.5093413166639429553953578980839738504498018104550134965469903497,
    .3072040820337873051847272159002587102563188179984871592895569129e-1,
    0,
  },
  { .1545084971874737120511467085914095294300772949514407155338621556,
    -.1545084971874737120511467085914095294300772949514407155338621556,
    0,
    1,
    0,
    -.3072040820337873051847272159002587102563188179984871592895569129e-1,
    .5093413166639429553953578980839738504498018104550134965469903497,
    .3072040820337873051847272159002587102563188179984871592895569129e-1,
    -.5093413166639429553953578980839738504498018104550134965469903497,
    0,
  },
  { .6545084971874737120511467085914095294300772949514407155338621555,
    .3454915028125262879488532914085904705699227050485592844661378439,
    0,
    0,
    .6530582494771754744398266018812554046880246378515774673927733836,
    -.3909998649235090370405005811858557276517203971888379362826572758e-1,
    -.3909998649235090370405005811858557276517203971888379362826572758e-1,
    .1437169328132325190444687037972815542382228273965639708457830349,
    .1437169328132325190444687037972815542382228273965639708457830349,
    -.8379578288972173185577336528559701739540157919035077699310042797e-2,
  },
  { 0,
    0,
    0,
    0,
    .6485558584258803742103469464952735558641901981510383445924168946e-1,
    .9728376155535236170776775142067550477777682216798278020248208813,
    .1871347032159210889457050798126409844727259829177195798876381701e-1,
    .3755195283204579196370815820910530342092739642135767508448271878e-1,
    -.8323901936022699853184835565403747248576980190586480879459775196e-2,
    -.8321806814020131213827501389514686274570541584411364509715675546e-3,
  },
  { 0,
    0,
    0,
    0,
    .6485558584258803742103469464952735558641901981510383445924168946e-1,
    .1871347032159210889457050798126409844727259829177195798876381701e-1,
    .9728376155535236170776775142067550477777682216798278020248208813,
    -.8323901936022699853184835565403747248576980190586480879459775196e-2,
    .3755195283204579196370815820910530342092739642135767508448271878e-1,
    -.8321806814020131213827501389514686274570541584411364509715675546e-3,
  },
  { 0,
    0,
    0,
    0,
    .2688827247825266957122321147715557742814783294657607374434182120,
    -.9607317289057197053957156255847105886192465697548737297297300402e-1,
    -.9607317289057197053957156255847105886192465697548737297297300402e-1,
    -.2104847253222102519577340299239342989252642376381788982316782848e-1,
    -.2104847253222102519577340299239342989252642376381788982316782848e-1,
    .9381176106433907444475179321313374063345690574299532587291526108e-2,
  },
  { 0,
    0,
    0,
    0,
    .9381176106433907444475179321313374063345690574299532587291526108e-2,
    -.3676093941456202108198900734694988307580073094903234388318636826e-1,
    -.3676093941456202108198900734694988307580073094903234388318636826e-1,
    .3826376094378892426180915221912774589359750226263713926661871225e-1,
    .3826376094378892426180915221912774589359750226263713926661871225e-1,
    .9998796274303411191767182438710709738452130516828383277311642554,
  },
  { 0,
    0,
    0,
    0,
    -.1819772312653810331563796602003918443408701005544090972553825527,
    .4530668710615203005918166624695813225157269130043427923282256455,
    -.1057274170411207591290343755909626814768710383713051707831755054e-2,
    .5177812083400424754778473064719315781588860877458726653878865325,
    -.2809464642802601633904568730257747251061828886607149057605530656e-1,
    .2335002210628916301202895011129331445253825472045931834404055334e-2,
  },
  { 0,
    0,
    0,
    0,
    -.1819772312653810331563796602003918443408701005544090972553825527,
    -.1057274170411207591290343755909626814768710383713051707831755054e-2,
    .4530668710615203005918166624695813225157269130043427923282256455,
    -.2809464642802601633904568730257747251061828886607149057605530656e-1,
    .5177812083400424754778473064719315781588860877458726653878872638,
    .2335002210628916301202895011129331445253825472045931834404055334e-2,
  }
};

/* >>> */
/* <<< interpolation matrices for child 1 */
/* Interpolation to child1 for type 0 and 1:
 *
 * Type 0:
 * 
 * lp = lc_0 e_1 + lc_2 e_2 + lc_1 e_3 + lc_3 1/2 (e_0 + e_1)
 *
 * lp_0 = 1/2 lc_3
 * lp_1 = lc_0 + 1/2 lc_3 = 1 - lc_1 - lc_2 - 1/2 lc_3
 * lp_2 = lc_2
 * lp_3 = lc_1
 */
static const REAL child1refine_inter[2][10][10] = {
  {
    { -.1545084971874737120511467085914095294300772949514407155338621556,
      1.154508497187473712051146708591409529430077294951440715533862155,
      0,
      0,
      -.6530582494771754744398266018812554046880246378515774673927733836,
      .3909998649235090370405005811858557276517203971888379362826573176e-1,
      .3909998649235090370405005811858557276517203971888379362826573176e-1,
      -.1437169328132325190444687037972815542382228273965639708457830335,
      -.1437169328132325190444687037972815542382228273965639708457830335,
      .8379578288972173185577336528559701739540157919035077699310042797e-2,
    },
    { -.1545084971874737120511467085914095294300772949514407155338621556,
      .1545084971874737120511467085914095294300772949514407155338621556,
      0,
      1,
      0,
      .3072040820337873051847272159002587102563188179984871592895569129e-1,
      -.5093413166639429553953578980839738504498018104550134965469903497,
      -.3072040820337873051847272159002587102563188179984871592895569129e-1,
      .5093413166639429553953578980839738504498018104550134965469903497,
      0,
    },
    { -.1545084971874737120511467085914095294300772949514407155338621556,
      .1545084971874737120511467085914095294300772949514407155338621556,
      1,
      0,
      0,
      -.5093413166639429553953578980839738504498018104550134965469903497,
      .3072040820337873051847272159002587102563188179984871592895569129e-1,
      .5093413166639429553953578980839738504498018104550134965469903497,
      -.3072040820337873051847272159002587102563188179984871592895569129e-1,
      0,
    },
    { .3454915028125262879488532914085904705699227050485592844661378439,
      .6545084971874737120511467085914095294300772949514407155338621555,
      0,
      0,
      .6530582494771754744398266018812554046880246378515774673927733836,
      .1437169328132325190444687037972815542382228273965639708457830335,
      .1437169328132325190444687037972815542382228273965639708457830335,
      -.3909998649235090370405005811858557276517203971888379362826572943e-1,
      -.3909998649235090370405005811858557276517203971888379362826572943e-1,
      -.8379578288972173185577336528559701739540157919035077699310042797e-2,
    },
    { 0,
      0,
      0,
      0,
      .6485558584258803742103469464952735558641901981510383445924168946e-1,
      -.8323901936022699853184835565403747248576980190586480879459774221e-2,
      .3755195283204579196370815820910530342092739642135767508448271878e-1,
      .1871347032159210889457050798126409844727259829177195798876381701e-1,
      .9728376155535236170776775142067550477777682216798278020248208813,
      -.8321806814020131213827501389514686274570541584411364509715675546e-3,
    },
    { 0,
      0,
      0,
      0,
      .6485558584258803742103469464952735558641901981510383445924168946e-1,
      .3755195283204579196370815820910530342092739642135767508448271878e-1,
      -.8323901936022699853184835565403747248576980190586480879459775196e-2,
      .9728376155535236170776775142067550477777682216798278020248208813,
      .1871347032159210889457050798126409844727259829177195798876381701e-1,
      -.8321806814020131213827501389514686274570541584411364509715675546e-3,
    },
    { 0,
      0,
      0,
      0,
      .2688827247825266957122321147715557742814783294657607374434182120,
      -.2104847253222102519577340299239342989252642376381788982316785041e-1,
      -.2104847253222102519577340299239342989252642376381788982316785041e-1,
      -.9607317289057197053957156255847105886192465697548737297297300402e-1,
      -.9607317289057197053957156255847105886192465697548737297297300402e-1,
      .9381176106433907444475179321313374063345690574299532587291526108e-2,
    },
    { 0,
      0,
      0,
      0,
      .9381176106433907444475179321313374063345690574299532587291526108e-2,
      .3826376094378892426180915221912774589359750226263713926661871225e-1,
      .3826376094378892426180915221912774589359750226263713926661871225e-1,
      -.3676093941456202108198900734694988307580073094903234388318636826e-1,
      -.3676093941456202108198900734694988307580073094903234388318636826e-1,
      .9998796274303411191767182438710709738452130516828383277311642554,
    },
    { 0,
      0,
      0,
      0,
      -.1819772312653810331563796602003918443408701005544090972553825527,
      -.2809464642802601633904568730257747251061828886607149057605530656e-1,
      .5177812083400424754778473064719315781588860877458726653878865325,
      -.1057274170411207591290343755909626814768710383713051707831755054e-2,
      .4530668710615203005918166624695813225157269130043427923282256455,
      .2335002210628916301202895011129331445253825472045931834404055334e-2,
    },
    { 0,
      0,
      0,
      0,
      -.1819772312653810331563796602003918443408701005544090972553825527,
      .5177812083400424754778473064719315781588860877458726653878872638,
      -.2809464642802601633904568730257747251061828886607149057605530656e-1,
      .4530668710615203005918166624695813225157269130043427923282256455,
      -.1057274170411207591290343755909626814768710383713051707831755054e-2,
      .2335002210628916301202895011129331445253825472045931834404055334e-2,
    },
  },
  /* lp = lc_0 e_1 + lc_1 e_2 + lc_2 e_3 + lc_3 1/2 (e_0 + e_1)
   *
   * lp_0 = 1/2 lc_3
   * lp_1 = lc_0 + 1/2 lc_3 = 1 - lc_1 - lc_2 - 1/2 lc_3
   * lp_2 = lc_1
   * lp_3 = lc_2
   */    
  {
    { -.1545084971874737120511467085914095294300772949514407155338621556,
      1.154508497187473712051146708591409529430077294951440715533862155,
      0,
      0,
      -.6530582494771754744398266018812554046880246378515774673927733836,
      .3909998649235090370405005811858557276517203971888379362826573270e-1,
      .3909998649235090370405005811858557276517203971888379362826573270e-1,
      -.1437169328132325190444687037972815542382228273965639708457830335,
      -.1437169328132325190444687037972815542382228273965639708457830335,
      .8379578288972173185577336528559701739540157919035077699310042797e-2,
    },
    { -.1545084971874737120511467085914095294300772949514407155338621556,
      .1545084971874737120511467085914095294300772949514407155338621556,
      1,
      0,
      0,
      -.5093413166639429553953578980839738504498018104550134965469903497,
      .3072040820337873051847272159002587102563188179984871592895569129e-1,
      .5093413166639429553953578980839738504498018104550134965469903497,
      -.3072040820337873051847272159002587102563188179984871592895569129e-1,
      0,
    },
    { -.1545084971874737120511467085914095294300772949514407155338621556,
      .1545084971874737120511467085914095294300772949514407155338621556,
      0,
      1,
      0,
      .3072040820337873051847272159002587102563188179984871592895569129e-1,
      -.5093413166639429553953578980839738504498018104550134965469903470,
      -.3072040820337873051847272159002587102563188179984871592895569129e-1,
      .5093413166639429553953578980839738504498018104550134965469903485,
      0,
    },
    { .3454915028125262879488532914085904705699227050485592844661378439,
      .6545084971874737120511467085914095294300772949514407155338621555,
      0,
      0,
      .6530582494771754744398266018812554046880246378515774673927733836,
      .1437169328132325190444687037972815542382228273965639708457830335,
      .1437169328132325190444687037972815542382228273965639708457830335,
      -.3909998649235090370405005811858557276517203971888379362826572943e-1,
      -.3909998649235090370405005811858557276517203971888379362826572943e-1,
      -.8379578288972173185577336528559701739540157919035077699310042797e-2,
    },
    { 0,
      0,
      0,
      0,
      .6485558584258803742103469464952735558641901981510383445924168946e-1,
      .3755195283204579196370815820910530342092739642135767508448271878e-1,
      -.8323901936022699853184835565403747248576980190586480879459774467e-2,
      .9728376155535236170776775142067550477777682216798278020248208813,
      .1871347032159210889457050798126409844727259829177195798876381701e-1,
      -.8321806814020131213827501389514686274570541584411364509715675546e-3,
    },
    { 0,
      0,
      0,
      0,
      .6485558584258803742103469464952735558641901981510383445924168946e-1,
      -.8323901936022699853184835565403747248576980190586480879459774221e-2,
      .3755195283204579196370815820910530342092739642135767508448271878e-1,
      .1871347032159210889457050798126409844727259829177195798876381701e-1,
      .9728376155535236170776775142067550477777682216798278020248208813,
      -.8321806814020131213827501389514686274570541584411364509715675546e-3,
    },
    { 0,
      0,
      0,
      0,
      .2688827247825266957122321147715557742814783294657607374434182120,
      -.2104847253222102519577340299239342989252642376381788982316785041e-1,
      -.2104847253222102519577340299239342989252642376381788982316785041e-1,
      -.9607317289057197053957156255847105886192465697548737297297300402e-1,
      -.9607317289057197053957156255847105886192465697548737297297300402e-1,
      .9381176106433907444475179321313374063345690574299532587291526108e-2,
    },
    { 0,
      0,
      0,
      0,
      .9381176106433907444475179321313374063345690574299532587291526108e-2,
      .3826376094378892426180915221912774589359750226263713926661871225e-1,
      .3826376094378892426180915221912774589359750226263713926661871225e-1,
      -.3676093941456202108198900734694988307580073094903234388318636826e-1,
      -.3676093941456202108198900734694988307580073094903234388318636826e-1,
      .9998796274303411191767182438710709738452130516828383277311642554,
    },
    { 0,
      0,
      0,
      0,
      -.1819772312653810331563796602003918443408701005544090972553825527,
      .5177812083400424754778473064719315781588860877458726653878872638,
      -.2809464642802601633904568730257747251061828886607149057605530656e-1,
      .4530668710615203005918166624695813225157269130043427923282256455,
      -.1057274170411207591290343755909626814768710383713051707831755054e-2,
      .2335002210628916301202895011129331445253825472045931834404055334e-2,
    },
    { 0,
      0,
      0,
      0,
      -.1819772312653810331563796602003918443408701005544090972553825527,
      -.2809464642802601633904568730257747251061828886607149057605530656e-1,
      .5177812083400424754778473064719315781588860877458726653878872638,
      -.1057274170411207591290343755909626814768710383713051707831755054e-2,
      .4530668710615203005918166624695813225157269130043427923282256455,
      .2335002210628916301202895011129331445253825472045931834404055334e-2,
    },
  }
};

/* >>> */

/* Interpolation during coarsening  */

/* <<< interpolation matrix for child 0 */
static const REAL child0coarse_inter[10][10] =
{ 
  { .5772542485937368560255733542957047647150386474757203577669310778,
    .7725424859373685602557335429570476471503864747572035776693107780e-1,
    .7725424859373685602557335429570476471503864747572035776693107780e-1,
    .3272542485937368560255733542957047647150386474757203577669310778,
    0,
    0,
    0,
    0,
    0,
    0,
  },
  { -.7725424859373685602557335429570476471503864747572035776693107780e-1,
    -.7725424859373685602557335429570476471503864747572035776693107780e-1,
    -.7725424859373685602557335429570476471503864747572035776693107780e-1,
    .1727457514062631439744266457042952352849613525242796422330689222,
    0,
    0,
    0,
    0,
    0,
    0,
  },
  { 0,
    .5000000000000000000000000000000000000000000000000000000000000000,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
  },
  { 0,
    0,
    .5000000000000000000000000000000000000000000000000000000000000000,
    0,
    0,
    0,
    0,
    0,
    0,
    0,
  },
  { -.3265291247385877372199133009406277023440123189257887336963866097,
    0,
    0,
    .3265291247385877372199133009406277023440123189257887336963866097,
    .3242779292129401871051734732476367779320950990755191722962084472e-1,
    .3242779292129401871051734732476367779320950990755191722962084472e-1,
    .1344413623912633478561160573857778871407391647328803687217093497,
    .4690588053216953722237589660656687031672845287149766293645765491e-2,
    -.9098861563269051657818983010019592217043505027720454862769127632e-1,
    -.9098861563269051657818983010019592217043505027720454862769127632e-1,
  },
  { -.7185846640661625952223435189864077711911141369828198542289149756e-1,
    .2546706583319714776976789490419869252249009052275067482734951122,
    -.1536020410168936525923636079501293551281594089992435796447784309e-1,
    -.1954999324617545185202502905929278638258601985944189681413285776e-1,
    .4864188077767618085388387571033775238888841108399139010124105627,
    .9356735160796054447285253990632049223636299145885978994381890230e-2,
    -.4803658644528598526978578127923552943096232848774368648648650812e-1,
    -.1838046970728101054099450367347494153790036547451617194159319144e-1,
    .2265334355307601502959083312347906612578634565021713961641128289,
    -.5286370852056037956451718779548134073843551918565258539158799643e-3,
  },
  { -.7185846640661625952223435189864077711911141369828198542289149756e-1,
    -.1536020410168936525923636079501293551281594089992435796447784309e-1,
    .2546706583319714776976789490419869252249009052275067482734951122,
    -.1954999324617545185202502905929278638258601985944189681413285776e-1,
    .9356735160796054447285253990632049223636299145885978994381890230e-2,
    .4864188077767618085388387571033775238888841108399139010124108063,
    -.4803658644528598526978578127923552943096232848774368648648650812e-1,
    -.1838046970728101054099450367347494153790036547451617194159319144e-1,
    -.5286370852056037956451718779548134073843551918565258539158799643e-3,
    .2265334355307601502959083312347906612578634565021713961641128289,
  },
  { .1954999324617545185202502905929278638258601985944189681413285776e-1,
    -.2546706583319714776976789490419869252249009052275067482734951122,
    .1536020410168936525923636079501293551281594089992435796447784309e-1,
    .7185846640661625952223435189864077711911141369828198542289149756e-1,
    .1877597641602289598185407910455265171046369821067883754224135939e-1,
    -.4161950968011349926592417782701873624288490095293240439729887965e-2,
    -.1052423626611051259788670149619671494626321188190894491158392399e-1,
    .1913188047189446213090457610956387294679875113131856963330936832e-1,
    .2588906041700212377389236532359657890794430438729363326939435101,
    -.1404732321401300816952284365128873625530914443303574528802765938e-1,
  },
  { .1954999324617545185202502905929278638258601985944189681413285776e-1,
    .1536020410168936525923636079501293551281594089992435796447784309e-1,
    -.2546706583319714776976789490419869252249009052275067482734951122,
    .7185846640661625952223435189864077711911141369828198542289149756e-1,
    -.4161950968011349926592417782701873624288490095293240439729887965e-2,
    .1877597641602289598185407910455265171046369821067883754224160310e-1,
    -.1052423626611051259788670149619671494626321188190894491158392399e-1,
    .1913188047189446213090457610956387294679875113131856963330936832e-1,
    -.1404732321401300816952284365128873625530914443303574528802765938e-1,
    .2588906041700212377389236532359657890794430438729363326939435101,
  },
  { .4189789144486086592788668264279850869770078959517538849655014053e-2,
    0,
    0,
    -.4189789144486086592788668264279850869770078959517538849655014053e-2,
    -.4160903407010065606913750694757343137285270792205682254857837772e-3,
    -.4160903407010065606913750694757343137285270792205682254857837772e-3,
    .4690588053216953722237589660656687031672845287149766293645765491e-2,
    .4999398137151705595883591219355354869226065258414191638655821153,
    .1167501105314458150601447505564665722626912736022965917202027667e-2,
    .1167501105314458150601447505564665722626912736022965917202027667e-2,
  }
};

/* >>> */
/* <<< interpolation matrices for child 1 */
static const REAL child1coarse_inter[2][10][10] =
{ {
    { -.7725424859373685602557335429570476471503864747572035776693107780e-1,
      -.7725424859373685602557335429570476471503864747572035776693107780e-1,
      -.7725424859373685602557335429570476471503864747572035776693107780e-1,
      .1727457514062631439744266457042952352849613525242796422330689222,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { .5772542485937368560255733542957047647150386474757203577669310778,
      .7725424859373685602557335429570476471503864747572035776693107780e-1,
      .7725424859373685602557335429570476471503864747572035776693107780e-1,
      .3272542485937368560255733542957047647150386474757203577669310778,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { 0,
      0,
      .5000000000000000000000000000000000000000000000000000000000000000,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { 0,
      .5000000000000000000000000000000000000000000000000000000000000000,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { -.3265291247385877372199133009406277023440123189257887336963866097,
      0,
      0,
      .3265291247385877372199133009406277023440123189257887336963866097,
      .3242779292129401871051734732476367779320950990755191722962084472e-1,
      .3242779292129401871051734732476367779320950990755191722962084472e-1,
      .1344413623912633478561160573857778871407391647328803687217097152,
      .4690588053216953722237589660656687031672845287149766293645765491e-2,
      -.9098861563269051657818983010019592217043505027720454862769127632e-1,
      -.9098861563269051657818983010019592217043505027720454862769127632e-1,
    },
    { .1954999324617545185202502905929278638258601985944189681413285776e-1,
      .1536020410168936525923636079501293551281594089992435796447784309e-1,
      -.2546706583319714776976789490419869252249009052275067482734951122,
      .7185846640661625952223435189864077711911141369828198542289149756e-1,
      -.4161950968011349926592417782701873624288490095293240439729887965e-2,
      .1877597641602289598185407910455265171046369821067883754224148125e-1,
      -.1052423626611051259788670149619671494626321188190894491158392399e-1,
      .1913188047189446213090457610956387294679875113131856963330936832e-1,
      -.1404732321401300816952284365128873625530914443303574528802765938e-1,
      .2588906041700212377389236532359657890794430438729363326939435101,
    },
    { .1954999324617545185202502905929278638258601985944189681413285776e-1,
      -.2546706583319714776976789490419869252249009052275067482734951122,
      .1536020410168936525923636079501293551281594089992435796447784309e-1,
      .7185846640661625952223435189864077711911141369828198542289149756e-1,
      .1877597641602289598185407910455265171046369821067883754224135939e-1,
      -.4161950968011349926592417782701873624288490095293240439729887965e-2,
      -.1052423626611051259788670149619671494626321188190894491158392399e-1,
      .1913188047189446213090457610956387294679875113131856963330936832e-1,
      .2588906041700212377389236532359657890794430438729363326939435101,
      -.1404732321401300816952284365128873625530914443303574528802765938e-1,
    },
    { -.7185846640661625952223435189864077711911141369828198542289149756e-1,
      -.1536020410168936525923636079501293551281594089992435796447784005e-1,
      .2546706583319714776976789490419869252249009052275067482734951122,
      -.1954999324617545185202502905929278638258601985944189681413285776e-1,
      .9356735160796054447285253990632049223636299145885978994381890230e-2,
      .4864188077767618085388387571033775238888841108399139010124105627,
      -.4803658644528598526978578127923552943096232848774368648648650812e-1,
      -.1838046970728101054099450367347494153790036547451617194159319144e-1,
      -.5286370852056037956451718779548134073843551918565258539158799643e-3,
      .2265334355307601502959083312347906612578634565021713961641128289,
    },
    { -.7185846640661625952223435189864077711911141369828198542289149756e-1,
      .2546706583319714776976789490419869252249009052275067482734951122,
      -.1536020410168936525923636079501293551281594089992435796447784492e-1,
      -.1954999324617545185202502905929278638258601985944189681413285776e-1,
      .4864188077767618085388387571033775238888841108399139010124105627,
      .9356735160796054447285253990632049223636299145885978994381890230e-2,
      -.4803658644528598526978578127923552943096232848774368648648650812e-1,
      -.1838046970728101054099450367347494153790036547451617194159319144e-1,
      .2265334355307601502959083312347906612578634565021713961641128289,
      -.5286370852056037956451718779548134073843551918565258539158799643e-3,
    },
    { .4189789144486086592788668264279850869770078959517538849655014053e-2,
      0,
      0,
      -.4189789144486086592788668264279850869770078959517538849655014053e-2,
      -.4160903407010065606913750694757343137285270792205682254857837772e-3,
      -.4160903407010065606913750694757343137285270792205682254857837772e-3,
      .4690588053216953722237589660656687031672845287149766293645765491e-2,
      .4999398137151705595883591219355354869226065258414191638655821153,
      .1167501105314458150601447505564665722626912736022965917202027667e-2,
      .1167501105314458150601447505564665722626912736022965917202027667e-2,
    }
  },
  {
    { -.7725424859373685602557335429570476471503864747572035776693107780e-1,
      -.7725424859373685602557335429570476471503864747572035776693107780e-1,
      -.7725424859373685602557335429570476471503864747572035776693107780e-1,
      .1727457514062631439744266457042952352849613525242796422330689222,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { .5772542485937368560255733542957047647150386474757203577669310778,
      .7725424859373685602557335429570476471503864747572035776693107780e-1,
      .7725424859373685602557335429570476471503864747572035776693107780e-1,
      .3272542485937368560255733542957047647150386474757203577669310778,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { 0,
      .5000000000000000000000000000000000000000000000000000000000000000,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { 0,
      0,
      .5000000000000000000000000000000000000000000000000000000000000000,
      0,
      0,
      0,
      0,
      0,
      0,
      0,
    },
    { -.3265291247385877372199133009406277023440123189257887336963866097,
      0,
      0,
      .3265291247385877372199133009406277023440123189257887336963866097,
      .3242779292129401871051734732476367779320950990755191722962084472e-1,
      .3242779292129401871051734732476367779320950990755191722962084472e-1,
      .1344413623912633478561160573857778871407391647328803687217093497,
      .4690588053216953722237589660656687031672845287149766293645765491e-2,
      -.9098861563269051657818983010019592217043505027720454862769127632e-1,
      -.9098861563269051657818983010019592217043505027720454862769127632e-1,
    },
    { .1954999324617545185202502905929278638258601985944189681413285776e-1,
      -.2546706583319714776976789490419869252249009052275067482734951122,
      .1536020410168936525923636079501293551281594089992435796447784309e-1,
      .7185846640661625952223435189864077711911141369828198542289149756e-1,
      .1877597641602289598185407910455265171046369821067883754224135939e-1,
      -.4161950968011349926592417782701873624288490095293240439729887965e-2,
      -.1052423626611051259788670149619671494626321188190894491158392399e-1,
      .1913188047189446213090457610956387294679875113131856963330936832e-1,
      .2588906041700212377389236532359657890794430438729363326939435101,
      -.1404732321401300816952284365128873625530914443303574528802765938e-1,
    },
    { .1954999324617545185202502905929278638258601985944189681413285776e-1,
      .1536020410168936525923636079501293551281594089992435796447784492e-1,
      -.2546706583319714776976789490419869252249009052275067482734951122,
      .7185846640661625952223435189864077711911141369828198542289149756e-1,
      -.4161950968011349926592417782701873624288490095293240439729887965e-2,
      .1877597641602289598185407910455265171046369821067883754224160310e-1,
      -.1052423626611051259788670149619671494626321188190894491158392399e-1,
      .1913188047189446213090457610956387294679875113131856963330936832e-1,
      -.1404732321401300816952284365128873625530914443303574528802765938e-1,
      .2588906041700212377389236532359657890794430438729363326939435101,
    },
    { -.7185846640661625952223435189864077711911141369828198542289149756e-1,
      .2546706583319714776976789490419869252249009052275067482734951122,
      -.1536020410168936525923636079501293551281594089992435796447783944e-1,
      -.1954999324617545185202502905929278638258601985944189681413285776e-1,
      .4864188077767618085388387571033775238888841108399139010124104408,
      .9356735160796054447285253990632049223636299145885978994381890230e-2,
      -.4803658644528598526978578127923552943096232848774368648648650812e-1,
      -.1838046970728101054099450367347494153790036547451617194159319144e-1,
      .2265334355307601502959083312347906612578634565021713961641128289,
      -.5286370852056037956451718779548134073843551918565258539158799643e-3,
    },
    { -.7185846640661625952223435189864077711911141369828198542289149756e-1,
      -.1536020410168936525923636079501293551281594089992435796447784309e-1,
      .2546706583319714776976789490419869252249009052275067482734951122,
      -.1954999324617545185202502905929278638258601985944189681413285776e-1,
      .9356735160796054447285253990632049223636299145885978994381890230e-2,
      .4864188077767618085388387571033775238888841108399139010124105627,
      -.4803658644528598526978578127923552943096232848774368648648650812e-1,
      -.1838046970728101054099450367347494153790036547451617194159319144e-1,
      -.5286370852056037956451718779548134073843551918565258539158799643e-3,
      .2265334355307601502959083312347906612578634565021713961641128289,
    },
    { .4189789144486086592788668264279850869770078959517538849655014053e-2,
      0,
      0,
      -.4189789144486086592788668264279850869770078959517538849655014053e-2,
      -.4160903407010065606913750694757343137285270792205682254857837772e-3,
      -.4160903407010065606913750694757343137285270792205682254857837772e-3,
      .4690588053216953722237589660656687031672845287149766293645765491e-2,
      .4999398137151705595883591219355354869226065258414191638655821153,
      .1167501105314458150601447505564665722626912736022965917202027667e-2,
      .1167501105314458150601447505564665722626912736022965917202027667e-2,
    }
  }
};
/* >>> */

/******************************************************************************
 *
 * Degree 1
 *
 ******************************************************************************/

#undef DEF_EL_VEC_D_ORTHO_1_3D
#define DEF_EL_VEC_D_ORTHO_1_3D(type, name)			\
  DEF_EL_VEC_CONST(type, name, N_BAS_LAG_1_3D, N_BAS_LAG_1_3D)

#undef DEFUN_GET_EL_VEC_D_ORTHO_1_3D
#define DEFUN_GET_EL_VEC_D_ORTHO_1_3D(name, type, admin, body, ...)	\
  static const EL_##type##_VEC *					\
  d_ortho_get_##name##_1_3d(type##_VEC_TYPE *vec, const EL *el, __VA_ARGS__) \
  {									\
    FUNCNAME("d_get_"#name"d_1_3d");					\
    static DEF_EL_VEC_D_ORTHO_1_3D(type, rvec_space);			\
    type##_VEC_TYPE *rvec = vec ? vec : rvec_space->vec;		\
    int n0, node, ibas;							\
    DOF **dofptr = el->dof, dof;					\
									\
    DEBUG_TEST_EXIT(true, "");						\
									\
    node = (admin)->mesh->node[CENTER];					\
    n0   = (admin)->n0_dof[CENTER];					\
    for (ibas = 0; ibas < N_BAS_LAG_1_3D; ibas++) {			\
      dof = dofptr[node][n0+ibas];					\
      body;								\
    }									\
									\
    return vec ? NULL : rvec_space;					\
  }									\
  struct _AI_semicolon_dummy

#undef DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D
#define DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(name, type, ASSIGN)		\
  DEFUN_GET_EL_VEC_D_ORTHO_1_3D(_##name##_vec, type, dv->fe_space->admin, \
				ASSIGN(dv->vec[dof], rvec[ibas]),	\
				const DOF_##type##_VEC *dv);		\
  static const EL_##type##_VEC *					\
  d_ortho_get_##name##_vec_1_3d(type##_VEC_TYPE *vec, const EL *el,	\
			       const DOF_##type##_VEC *dv)		\
  {									\
    EL_##type##_VEC *vec_loc = dv->vec_loc;				\
									\
    if (vec != NULL || vec_loc == NULL) {				\
      return d_ortho_get__##name##_vec_1_3d(vec, el, dv);		\
    } else {								\
      d_ortho_get__##name##_vec_1_3d(vec_loc->vec, el, dv);		\
      return vec_loc;							\
    }									\
  }									\
  struct _AI_semicolon_dummy

#undef COPY_EQ
#define COPY_EQ(a, b) (b) = (a)

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

DEFUN_GET_EL_VEC_D_ORTHO_1_3D(dof_indices, DOF, admin,
			      rvec[ibas] = dof,
			      const DOF_ADMIN *admin, const BAS_FCTS *thisptr);

static const EL_BNDRY_VEC *d_ortho_get_bound_1_3d(BNDRY_FLAGS *vec,
						  const EL_INFO *el_info,
						  const BAS_FCTS *thisptr)
{
  FUNCNAME("d_get_bound2_3d");
  DEF_EL_VEC_D_ORTHO_1_3D(BNDRY, rvec_space);
  BNDRY_FLAGS *rvec = vec ? vec : rvec_space->vec;
  int i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_BAS_LAG_1_3D; i++) {
    BNDRY_FLAGS_INIT(rvec[i]);
  }

  return vec ? NULL : rvec_space;
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_INT_VEC                   ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(int, INT, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_VEC                  ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(real, REAL, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_D_VEC                ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(real_d, REAL_D, COPY_DOW);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_SCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(schar, SCHAR, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_UCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(uchar, UCHAR, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_PTR_VEC                   ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(ptr, PTR, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_DD_VEC               ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_1_3D(real_dd, REAL_DD, _AI_MCOPY_DOW);

/****************************************************************************/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/****************************************************************************/

/* We have 
 */
static void
d_ortho_real_refine_inter_1_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  EL *child;  
  int i, j, k, n0, node;

  n0 = drv->fe_space->admin->n0_dof[CENTER];
  node = drv->fe_space->admin->mesh->node[CENTER];

  for (k = 0; k < n; k++) {
    EL *el = list[k].el_info.el;
    int el_type = list[k].el_info.el_type;
    
    /* child0, all types */    
    child = el->child[0];
    for (i = 0; i < N_BAS_LAG_1_3D; i++) {
      DOF cdof = child->dof[node][n0+i];
      drv->vec[cdof] = 0.0;
      for (j = 0; j < N_BAS_LAG_1_3D; j++) {
	DOF pdof = el->dof[node][n0+j];	
	drv->vec[cdof] += child0refine_inter[i][j] * drv->vec[pdof];
      }
    }
    {
      int v;
      
      for (v = 0; v < N_VERTICES_3D; v++) {
	REAL_B lambda;
	REAL value_c, value_p;
	
	SET_BAR(3, 0.0, lambda);
	lambda[v] = 1.0;
	value_c = 0.0;
	for (i = 0; i < N_BAS_LAG_1_3D; i++) {
	  DOF cdof = child->dof[node][n0+i];
	  value_c +=  drv->vec[cdof] * d_ortho_phi_v_3d(lambda, i);
	}
	SET_BAR(3, 0.0, lambda);
	if (v < N_VERTICES_3D-1) {
	  lambda[child_vertex_3d[0][0][v]] = 1.0;
	} else {
	  lambda[0] = lambda[1] = 0.5;
	}
	value_p = 0.0;
	for (i = 0; i < N_BAS_LAG_1_3D; i++) {
	  DOF pdof = el->dof[node][n0+i];
	  value_p +=  drv->vec[pdof] * d_ortho_phi_v_3d(lambda, i);
	}
	TEST_EXIT(fabs(value_p - value_c) < 1e-10, "interpolation weirdness\n");
      }
    }



    /* child1, type dependent */
    child = el->child[1];
    for (i = 0; i < N_BAS_LAG_1_3D; i++) {
      DOF cdof = child->dof[node][n0+i];
      drv->vec[cdof] = 0.0;
      for (j = 0; j < N_BAS_LAG_1_3D; j++) {
	DOF pdof = el->dof[node][n0+j];	
	drv->vec[cdof] +=
	  child1refine_inter[el_type > 0][i][j] * drv->vec[pdof];
      }
    }

    {
      int v;
      
      for (v = 0; v < N_VERTICES_3D; v++) {
	REAL_B lambda;
	REAL value_c, value_p;
	
	SET_BAR(3, 0.0, lambda);
	lambda[v] = 1.0;
	value_c = 0.0;
	for (i = 0; i < N_BAS_LAG_1_3D; i++) {
	  DOF cdof = child->dof[node][n0+i];
	  value_c +=  drv->vec[cdof] * d_ortho_phi_v_3d(lambda, i);
	}
	SET_BAR(3, 0.0, lambda);
	if (v < N_VERTICES_3D-1) {
	  lambda[child_vertex_3d[el_type][1][v]] = 1.0;
	} else {
	  lambda[0] = lambda[1] = 0.5;
	}
	value_p = 0.0;
	for (i = 0; i < N_BAS_LAG_1_3D; i++) {
	  DOF pdof = el->dof[node][n0+i];
	  value_p +=  drv->vec[pdof] * d_ortho_phi_v_3d(lambda, i);
	}
	TEST_EXIT(fabs(value_p - value_c) < 1e-10, "interpolation weirdness\n");
      }
    }



  }
}

static void
d_ortho_real_coarse_inter_1_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  EL *child;  
  int i, j, k, n0, node;

  n0 = drv->fe_space->admin->n0_dof[CENTER];
  node = drv->fe_space->admin->mesh->node[CENTER];

  for (k = 0; k < n; k++) {
    EL *el = list[k].el_info.el;
    int el_type = list[k].el_info.el_type;
    
    /* child0, all types */    
    child = el->child[0];
    for (i = 0; i < N_BAS_LAG_1_3D; i++) {
      DOF pdof = el->dof[node][n0+i];
      drv->vec[pdof] = 0.0;
      for (j = 0; j < N_BAS_LAG_1_3D; j++) {
	DOF cdof = child->dof[node][n0+j];	
	drv->vec[pdof] += child0coarse_inter[i][j] * drv->vec[cdof];
      }
    }

    /* child1, type dependent */
    child = el->child[1];
    for (i = 0; i < N_BAS_LAG_1_3D; i++) {
      DOF pdof = el->dof[node][n0+i];
      for (j = 0; j < N_BAS_LAG_1_3D; j++) {
	DOF cdof = child->dof[node][n0+j];	
	drv->vec[pdof] +=
	  child1coarse_inter[el_type > 0][i][j] * drv->vec[cdof];
      }
    }
  }
}

/* Interpolation: we always use L2-interpolation over the entire
 * element, even if WALL >= 0. The quadrature degree is such that we
 * would reproduce the polynomials of the given degree.
 */
static void
d_ortho_interpol_1_3d(EL_REAL_VEC *el_vec,
		      const EL_INFO *el_info, int wall,
		      int no, const int *b_no,
		      LOC_FCT_AT_QP f, void *f_data,
		      const BAS_FCTS *thisptr)
{
  const QUAD_FAST *qfast = ((ORTHO_DATA *)thisptr->ext_data)->qfast;
  
  if (b_no) {
    int i, iq;
    
    for (i = 0; i < no; i++) {
      el_vec->vec[b_no[i]] = 0.0;      
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL value = qfast->w[iq] * f(el_info, qfast->quad, iq, f_data);
      for (i = 0; i < no; i++) {
	int ib = b_no[i];
	el_vec->vec[ib] += qfast->phi[iq][ib] * value;
      }
    }
  } else {
    int iq, ib;

    for (ib = 0; ib < N_BAS_LAG_1_3D; ib++) {
      el_vec->vec[ib] = 0.0;
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL value = qfast->w[iq] * f(el_info, qfast->quad, iq, f_data);
      for (ib = 0; ib < N_BAS_LAG_1_3D; ib++) {
	el_vec->vec[ib] += qfast->phi[iq][ib] * value;
      }
    }
  }
}

static void
d_ortho_interpol_d_1_3d(EL_REAL_D_VEC *el_vec,
			const EL_INFO *el_info, int wall,
			int no, const int *b_no,
			LOC_FCT_D_AT_QP f, void *f_data,
			const BAS_FCTS *thisptr)
{
  const QUAD_FAST *qfast = ((ORTHO_DATA *)thisptr->ext_data)->qfast;
  
  if (b_no) {
    int i, iq;
    
    for (i = 0; i < no; i++) {
      SET_DOW(0.0, el_vec->vec[b_no[i]]);
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL_D value;

      f(value, el_info, qfast->quad, iq, f_data);
      SCAL_DOW(qfast->w[iq], value);
      for (i = 0; i < no; i++) {
	int ib = b_no[i];
	AXPY_DOW( qfast->phi[iq][ib], value, el_vec->vec[ib]);
      }
    }
  } else {
    int iq, ib;

    for (ib = 0; ib < N_BAS_LAG_1_3D; ib++) {
      SET_DOW(0.0, el_vec->vec[ib]);
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL_D value;

      f(value, el_info, qfast->quad, iq, f_data);
      SCAL_DOW(qfast->w[iq], value);
      for (ib = 0; ib < N_BAS_LAG_1_3D; ib++) {
	AXPY_DOW(qfast->phi[iq][ib], value, el_vec->vec[ib]);
      }
    }
  }
}

static void
d_ortho_interpol_dow_1_3d(EL_REAL_VEC_D *el_vec,
			  const EL_INFO *el_info, int wall,
			  int no, const int *b_no,
			  LOC_FCT_D_AT_QP f, void *f_data,
			  const BAS_FCTS *thisptr)
{
  d_ortho_interpol_d_1_3d((EL_REAL_D_VEC *)el_vec,
			  el_info, wall, no, b_no, f, f_data, thisptr);
}

/* The trace-space is degenerated: all basis functions have non-zero
 * trace on all walls. If permute the stuff cyclic s.t. we would
 * eventually have the chance to really define the trace-space as
 * degenerated set of basis functions.
 */
static const int trace_mapping_d_ortho_1_3d[N_WALLS_3D][N_BAS_LAG_1_3D] = {
  { 1, 2, 3, 0 }, { 0, 2, 3, 1 }, { 0, 1, 3, 2 }, { 0, 1, 2, 3 }
};

static const BAS_FCTS disc_ortho1_3d = {
  DISC_ORTHO_NAME(1)"_3d", 3, 1, N_BAS_LAG_1_3D, N_BAS_LAG_1_3D, 1,
  {0, N_BAS_LAG_1_3D, 0, 0},
  -1, /* trace_admin */
  INIT_BFCTS_CHAIN(disc_ortho1_3d),
  INIT_ELEMENT_INITIALIZER(NULL, FILL_NOTHING), /* init_element + fill-flags */
  d_ortho_phi_3d, d_ortho_grd_phi_3d, d_ortho_D2_phi_3d,
  NULL, NULL, /* third and fourth derivatives */
  NULL, NULL, NULL, false, /* phi_d etc. */  
  /********************/
  NULL, /* trace space */
  { { { trace_mapping_d_ortho_1_3d[0],
	trace_mapping_d_ortho_1_3d[1],
	trace_mapping_d_ortho_1_3d[2],
	trace_mapping_d_ortho_1_3d[3] },
      { trace_mapping_d_ortho_1_3d[0],
	trace_mapping_d_ortho_1_3d[1],
	trace_mapping_d_ortho_1_3d[2],
	trace_mapping_d_ortho_1_3d[3] } }, 
    { { trace_mapping_d_ortho_1_3d[0],
	trace_mapping_d_ortho_1_3d[1],
	trace_mapping_d_ortho_1_3d[2],
	trace_mapping_d_ortho_1_3d[3] },
      { trace_mapping_d_ortho_1_3d[0],
	trace_mapping_d_ortho_1_3d[1],
	trace_mapping_d_ortho_1_3d[2],
	trace_mapping_d_ortho_1_3d[3] } } }, /* trace mapping */
  { N_BAS_LAG_1_3D,
    N_BAS_LAG_1_3D,
    N_BAS_LAG_1_3D,
    N_BAS_LAG_1_3D, }, /* n_trace_bas_fcts */
  d_ortho_get_dof_indices_1_3d,
  d_ortho_get_bound_1_3d,
  d_ortho_interpol_1_3d,
  d_ortho_interpol_d_1_3d,
  d_ortho_interpol_dow_1_3d,
  d_ortho_get_int_vec_1_3d,
  d_ortho_get_real_vec_1_3d,
  d_ortho_get_real_d_vec_1_3d,
  (GET_REAL_VEC_D_TYPE)d_ortho_get_real_d_vec_1_3d,
  d_ortho_get_uchar_vec_1_3d,
  d_ortho_get_schar_vec_1_3d,
  d_ortho_get_ptr_vec_1_3d,
  d_ortho_get_real_dd_vec_1_3d,
  d_ortho_real_refine_inter_1_3d,
  d_ortho_real_coarse_inter_1_3d,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  (void *)&d_ortho_1_3d_data
};

/******************************************************************************
 *
 * Degree 2
 *
 ******************************************************************************/

#undef DEF_EL_VEC_D_ORTHO_2_3D
#define DEF_EL_VEC_D_ORTHO_2_3D(type, name)			\
  DEF_EL_VEC_CONST(type, name, N_BAS_LAG_2_3D, N_BAS_LAG_2_3D)

#undef DEFUN_GET_EL_VEC_D_ORTHO_2_3D
#define DEFUN_GET_EL_VEC_D_ORTHO_2_3D(name, type, admin, body, ...)	\
  static const EL_##type##_VEC *					\
  d_ortho_get_##name##_2_3d(type##_VEC_TYPE *vec, const EL *el, __VA_ARGS__) \
  {									\
    FUNCNAME("d_get_"#name"d_2_3d");					\
    static DEF_EL_VEC_D_ORTHO_2_3D(type, rvec_space);			\
    type##_VEC_TYPE *rvec = vec ? vec : rvec_space->vec;		\
    int n0, node, ibas;							\
    DOF **dofptr = el->dof, dof;					\
									\
    DEBUG_TEST_EXIT(true, "");						\
									\
    node = (admin)->mesh->node[CENTER];					\
    n0   = (admin)->n0_dof[CENTER];					\
    for (ibas = 0; ibas < N_BAS_LAG_2_3D; ibas++) {			\
      dof = dofptr[node][n0+ibas];					\
      body;								\
    }									\
									\
    return vec ? NULL : rvec_space;					\
  }									\
  struct _AI_semicolon_dummy

#undef DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D
#define DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(name, type, ASSIGN)		\
  DEFUN_GET_EL_VEC_D_ORTHO_2_3D(_##name##_vec, type, dv->fe_space->admin, \
				ASSIGN(dv->vec[dof], rvec[ibas]),	\
				const DOF_##type##_VEC *dv);		\
  static const EL_##type##_VEC *					\
  d_ortho_get_##name##_vec_2_3d(type##_VEC_TYPE *vec, const EL *el,	\
			       const DOF_##type##_VEC *dv)		\
  {									\
    EL_##type##_VEC *vec_loc = dv->vec_loc;				\
									\
    if (vec != NULL || vec_loc == NULL) {				\
      return d_ortho_get__##name##_vec_2_3d(vec, el, dv);		\
    } else {								\
      d_ortho_get__##name##_vec_2_3d(vec_loc->vec, el, dv);		\
      return vec_loc;							\
    }									\
  }									\
  struct _AI_semicolon_dummy

#undef COPY_EQ
#define COPY_EQ(a, b) (b) = (a)

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

DEFUN_GET_EL_VEC_D_ORTHO_2_3D(dof_indices, DOF, admin,
			      rvec[ibas] = dof,
			      const DOF_ADMIN *admin, const BAS_FCTS *thisptr);

static const EL_BNDRY_VEC *d_ortho_get_bound_2_3d(BNDRY_FLAGS *vec,
						  const EL_INFO *el_info,
						  const BAS_FCTS *thisptr)
{
  FUNCNAME("d_get_bound2_3d");
  DEF_EL_VEC_D_ORTHO_2_3D(BNDRY, rvec_space);
  BNDRY_FLAGS *rvec = vec ? vec : rvec_space->vec;
  int i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_BAS_LAG_2_3D; i++) {
    BNDRY_FLAGS_INIT(rvec[i]);
  }

  return vec ? NULL : rvec_space;
}

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_INT_VEC                   ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(int, INT, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_VEC                  ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(real, REAL, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_D_VEC                ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(real_d, REAL_D, COPY_DOW);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_SCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(schar, SCHAR, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_UCHAR_VEC                 ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(uchar, UCHAR, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_PTR_VEC                   ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(ptr, PTR, COPY_EQ);

/*--------------------------------------------------------------------*/
/*--- function for accessing a local DOF_REAL_DD_VEC               ---*/
/*--------------------------------------------------------------------*/

DEFUN_GET_EL_DOF_VEC_D_ORTHO_2_3D(real_dd, REAL_DD, _AI_MCOPY_DOW);

/****************************************************************************/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/****************************************************************************/

static void
d_ortho_real_refine_inter_2_3d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  EL *child;  
  int i, j, k, n0, node;

  n0 = drv->fe_space->admin->n0_dof[CENTER];
  node = drv->fe_space->admin->mesh->node[CENTER];

  for (k = 0; k < n; k++) {
    EL *el = list[k].el_info.el;
    int el_type = list[k].el_info.el_type;
    
    /* child0, all types */    
    child = el->child[0];
    for (i = 0; i < N_BAS_LAG_2_3D; i++) {
      DOF cdof = child->dof[node][n0+i];
      drv->vec[cdof] = 0.0;
      for (j = 0; j < N_BAS_LAG_2_3D; j++) {
	DOF pdof = el->dof[node][n0+j];	
	drv->vec[cdof] += child0refine_inter[i][j] * drv->vec[pdof];
      }
    }

    /* child1, type dependent */
    child = el->child[1];
    for (i = 0; i < N_BAS_LAG_2_3D; i++) {
      DOF cdof = child->dof[node][n0+i];
      drv->vec[cdof] = 0.0;
      for (j = 0; j < N_BAS_LAG_2_3D; j++) {
	DOF pdof = el->dof[node][n0+j];	
	drv->vec[cdof] +=
	  child1refine_inter[el_type > 0][i][j] * drv->vec[pdof];
      }
    }
  }
}

static void d_ortho_real_coarse_inter_2_3d(DOF_REAL_VEC *drv,
					   RC_LIST_EL *list, int n)
{
  EL *child;  
  int i, j, k, n0, node;

  n0 = drv->fe_space->admin->n0_dof[CENTER];
  node = drv->fe_space->admin->mesh->node[CENTER];

  for (k = 0; k < n; k++) {
    EL *el = list[k].el_info.el;
    int el_type = list[k].el_info.el_type;
    
    /* child0, all types */    
    child = el->child[0];
    for (i = 0; i < N_BAS_LAG_2_3D; i++) {
      DOF pdof = el->dof[node][n0+i];
      drv->vec[pdof] = 0.0;
      for (j = 0; j < N_BAS_LAG_2_3D; j++) {
	DOF cdof = child->dof[node][n0+j];	
	drv->vec[pdof] += child0coarse_inter[i][j] * drv->vec[cdof];
      }
    }

    /* child1, type dependent */
    child = el->child[1];
    for (i = 0; i < N_BAS_LAG_2_3D; i++) {
      DOF pdof = el->dof[node][n0+i];
      for (j = 0; j < N_BAS_LAG_2_3D; j++) {
	DOF cdof = child->dof[node][n0+j];	
	drv->vec[pdof] +=
	  child1coarse_inter[el_type > 0][i][j] * drv->vec[cdof];
      }
    }
  }
}

#undef SQRT2
#undef SQRT6
#undef SQRT10
#undef SQRT15
#undef DENOM
#undef DENOM2
#undef COEFF11
#undef COEFF12
#undef COEFF13
#undef COEFF14

/* Interpolation: we always use L2-interpolation over the entire
 * element, even if WALL >= 0. The quadrature degree is such that we
 * would reproduce the polynomials of the given degree.
 */
static void
d_ortho_interpol_2_3d(EL_REAL_VEC *el_vec,
		      const EL_INFO *el_info, int wall,
		      int no, const int *b_no,
		      LOC_FCT_AT_QP f, void *f_data,
		      const BAS_FCTS *thisptr)
{
  const QUAD_FAST *qfast = ((ORTHO_DATA *)thisptr->ext_data)->qfast;
  
  if (b_no) {
    int i, iq;
    
    for (i = 0; i < no; i++) {
      el_vec->vec[b_no[i]] = 0.0;      
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL value = qfast->w[iq] * f(el_info, qfast->quad, iq, f_data);
      for (i = 0; i < no; i++) {
	int ib = b_no[i];
	el_vec->vec[ib] += qfast->phi[iq][ib] * value;
      }
    }
  } else {
    int iq, ib;

    for (ib = 0; ib < N_BAS_LAG_2_3D; ib++) {
      el_vec->vec[ib] = 0.0;
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL value = qfast->w[iq] * f(el_info, qfast->quad, iq, f_data);
      for (ib = 0; ib < N_BAS_LAG_2_3D; ib++) {
	el_vec->vec[ib] += qfast->phi[iq][ib] * value;
      }
    }
  }
}

static void
d_ortho_interpol_d_2_3d(EL_REAL_D_VEC *el_vec,
			const EL_INFO *el_info, int wall,
			int no, const int *b_no,
			LOC_FCT_D_AT_QP f, void *f_data,
			const BAS_FCTS *thisptr)
{
  const QUAD_FAST *qfast = ((ORTHO_DATA *)thisptr->ext_data)->qfast;
  
  if (b_no) {
    int i, iq;
    
    for (i = 0; i < no; i++) {
      SET_DOW(0.0, el_vec->vec[b_no[i]]);
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL_D value;

      f(value, el_info, qfast->quad, iq, f_data);
      SCAL_DOW(qfast->w[iq], value);
      for (i = 0; i < no; i++) {
	int ib = b_no[i];
	AXPY_DOW( qfast->phi[iq][ib], value, el_vec->vec[ib]);
      }
    }
  } else {
    int iq, ib;

    for (ib = 0; ib < N_BAS_LAG_2_3D; ib++) {
      SET_DOW(0.0, el_vec->vec[ib]);
    }
    for (iq = 0; iq < qfast->n_points; iq++) {
      REAL_D value;

      f(value, el_info, qfast->quad, iq, f_data);
      SCAL_DOW(qfast->w[iq], value);
      for (ib = 0; ib < N_BAS_LAG_2_3D; ib++) {
	AXPY_DOW(qfast->phi[iq][ib], value, el_vec->vec[ib]);
      }
    }
  }
}

static void
d_ortho_interpol_dow_2_3d(EL_REAL_VEC_D *el_vec,
			  const EL_INFO *el_info, int wall,
			  int no, const int *b_no,
			  LOC_FCT_D_AT_QP f, void *f_data,
			  const BAS_FCTS *thisptr)
{
  d_ortho_interpol_d_1_3d((EL_REAL_D_VEC *)el_vec,
			  el_info, wall, no, b_no, f, f_data, thisptr);
}

/* The trace-space is degenerated: all basis functions have non-zero
 * trace on all walls. We just use an arbitrary ordering of the basis
 * functions.
 */
static const int trace_mapping_d_ortho_2_3d[N_WALLS_3D][N_BAS_LAG_2_3D] = {
  { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 },
  { 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }
};


static const BAS_FCTS disc_ortho2_3d = {
  DISC_ORTHO_NAME(2)"_3d", 3, 1, N_BAS_LAG_2_3D, N_BAS_LAG_2_3D, 2,
  {0, N_BAS_LAG_2_3D, 0, 0},
  -1, /* trace_admin */
  INIT_BFCTS_CHAIN(disc_ortho2_3d),
  INIT_ELEMENT_INITIALIZER(NULL, FILL_NOTHING), /* init_element + fill-flags */
  d_ortho_phi_3d, d_ortho_grd_phi_3d, d_ortho_D2_phi_3d,
  NULL, NULL, /* third and fourth derivatives */
  NULL, NULL, NULL, false, /* phi_d etc. */  
  /********************/
  NULL /* &disc_lagrange2_1d */, /* trace space */
  { { { trace_mapping_d_ortho_2_3d[0],
	trace_mapping_d_ortho_2_3d[1],
	trace_mapping_d_ortho_2_3d[2],
	trace_mapping_d_ortho_2_3d[3] },
      { trace_mapping_d_ortho_2_3d[0],
	trace_mapping_d_ortho_2_3d[1],
	trace_mapping_d_ortho_2_3d[2],
	trace_mapping_d_ortho_2_3d[3] } },
    { { trace_mapping_d_ortho_2_3d[0],
	trace_mapping_d_ortho_2_3d[1],
	trace_mapping_d_ortho_2_3d[2],
	trace_mapping_d_ortho_2_3d[3] },
      { trace_mapping_d_ortho_2_3d[0],
	trace_mapping_d_ortho_2_3d[1],
	trace_mapping_d_ortho_2_3d[2],
	trace_mapping_d_ortho_2_3d[3] } } }, /* trace mapping */
  { N_BAS_LAG_2_3D,
    N_BAS_LAG_2_3D,
    N_BAS_LAG_2_3D,
    N_BAS_LAG_2_3D, }, /* n_trace_bas_fcts */
  d_ortho_get_dof_indices_2_3d,
  d_ortho_get_bound_2_3d,
  d_ortho_interpol_2_3d,
  d_ortho_interpol_d_2_3d,
  d_ortho_interpol_dow_2_3d,
  d_ortho_get_int_vec_2_3d,
  d_ortho_get_real_vec_2_3d,
  d_ortho_get_real_d_vec_2_3d,
  (GET_REAL_VEC_D_TYPE)d_ortho_get_real_d_vec_2_3d,
  d_ortho_get_uchar_vec_2_3d,
  d_ortho_get_schar_vec_2_3d,
  d_ortho_get_ptr_vec_2_3d,
  d_ortho_get_real_dd_vec_2_3d,
  d_ortho_real_refine_inter_2_3d,
  d_ortho_real_coarse_inter_2_3d,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  NULL,
  (void *)&d_ortho_2_3d_data
};

