--  Copyright (C) 2002-2004 David Roundy
--
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; either version 2, or (at your option)
--  any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; see the file COPYING.  If not, write to
--  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
--  Boston, MA 02110-1301, USA.

module Darcs.UI.Commands.Util.Tree
    ( 
    -- * Tree lookup.
      treeHas
    , treeHasDir
    , treeHasFile
    , treeHasAnycase
    ) where

import Darcs.Prelude

import Data.List ( find )

import Darcs.Util.Path ( AnchoredPath(..), eqAnycase )
import Darcs.Util.Tree ( Tree, TreeItem(..), listImmediate )
import qualified Darcs.Util.Tree.Monad as TM
    ( directoryExists
    , exists
    , fileExists
    , virtualTreeMonad
    )

treeHasAnycase :: Monad m
               => Tree m
               -> AnchoredPath
               -> m Bool
treeHasAnycase tree (AnchoredPath names) = go names (SubTree tree)
  where
    go [] _ = return True
    go ns (Stub mkTree _) = mkTree >>= go ns . SubTree
    go _ (File _) = return False
    go (n:ns) (SubTree t) =
      case find (eqAnycase n . fst) (listImmediate t) of
        Nothing -> return False
        Just (_,i) -> go ns i

treeHas :: Monad m => Tree m -> AnchoredPath -> m Bool
treeHas tree path = fst `fmap` TM.virtualTreeMonad (TM.exists path) tree

treeHasDir :: Monad m => Tree m -> AnchoredPath -> m Bool
treeHasDir tree path = fst `fmap` TM.virtualTreeMonad (TM.directoryExists path) tree

treeHasFile :: Monad m => Tree m -> AnchoredPath -> m Bool
treeHasFile tree path = fst `fmap` TM.virtualTreeMonad (TM.fileExists path) tree
