/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifndef KBINDINGS_COMMON_H
#define KBINDINGS_COMMON_H

#include "kbindings.h"

/*
 * Key binding entry struct.
 */
typedef struct {
  const char       *name;        /** << Human readable action, used in config file too.
                                  * We currently only use the known ones, and thus can stick
                                  * to our default static const strings. */
  const char       *comment;     /** << Human readable description. */
  const char       *key;         /** << Key binding. */
  uint16_t          id;          /** << The numerical action id. */
  uint16_t          index;       /** << Into kbinding_t.entry. */
  uint8_t           modifier;    /** << Modifier key of binding (MODIFIER_*, can be OR'ed). */
#define KBE_FLAG_alias   1       /** << Optional alternative binding. */
#define KBE_FLAG_gui     2       /** << controls the gui (eg opening a dialogue) rather than the xine playback. */
#define KBE_FLAG_default 4       /** << Still or again the preset binding. */
#define KBE_FLAG_void    8       /** << no key bound to this one. */
  uint8_t           flags;
} kbinding_entry_t;

#define KBT_MAX_ENTRIES 320

kbinding_t *kbindings_dup (kbinding_t *kbt);

const kbinding_entry_t *kbindings_entry_from_name (kbinding_t *kbt, const char *name_or_id);
const kbinding_entry_t *kbindings_entry_from_index (kbinding_t *kbt, int index);
int kbindings_get_num_entries (kbinding_t *kbt);

/** key == "void" means delete.
 *  use ~index to add an alias, or key == NULL to reset default (index == -1 means "all" then).
 *  return -1 (OK), -2 (unchanged), -3 (invalid), -4 (table full), >= 0 (index that already uses this key). */
int kbindings_entry_set (kbinding_t *kbt, int index, int modifier, const char *key);
/** will also try (kbt, key, 0) if (kbt, key, modifier) is not found. */
kbinding_entry_t *kbindings_entry_from_key (kbinding_t *kbt, const char *key, int modifier);

const char * const *kbindings_modifier_names (kbinding_t *kbt);

#endif
