/* XRACER (C) 1999-2000 Richard W.M. Jones <rich@annexia.org> and other AUTHORS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Id: flare.c,v 1.3 2000/03/20 21:08:57 rich Exp $
 */

/* A lens flare effect, taken from a program by Mark Kilgard
 * <mjk@sgi.com> and available here:
 *
 * http://reality.sgi.com/opengl/tips/lensflare/
 *
 * Mark Kilgard in fact based his GL implementation on an
 * original Direct3D implementation written by Stephen Coy
 * at Microsoft. For reference, the original Direct3D
 * implementation ran to 12,000 lines of code. Wow :-)
 *
 * I have added and changed quite a few things.
 *                                              - Rich.
 */

#include "config.h"

#include <stdio.h>

#include <GL/gl.h>

#include "xracer.h"
#include "xracer-track.h"
#include "xracer-log.h"
#include "xracer-texture.h"
#include "common.h"

#define FLARE_SCALE_FACTOR 1.5

static GLuint shine_tex[10], flare_tex[6];

struct flare
{
  int type;			/* -1 == shine; 0 .. 5 = flare texture index */
  GLfloat scale;
  GLfloat loc;
  GLfloat colour[3];
};

#define NUM_FLARES 12
static struct flare flare[NUM_FLARES];

#define red xrColourRed
#define green xrColourGreen
#define blue xrColourBlue

/* XXX I suppose that this ought to match the general ambient
 * scene lighting. Find out what that is and fix it.
 */
static GLfloat sun_direction[3] = { 100, 0, 20 };

static struct flare
set_flare (int type, GLfloat location, GLfloat scale, const GLfloat *colour,
	   GLfloat colour_scale)
{
  struct flare r;

  r.type = type;
  r.loc = location;
  r.scale = scale;
  r.colour[0] = colour[0] * colour_scale;
  r.colour[1] = colour[1] * colour_scale;
  r.colour[2] = colour[2] * colour_scale;

  return r;
}

int
flare_load ()
{
  int i;
  char buffer[256];

  xrLog (LOG_DEBUG, "Loading shine and flare textures");

  /* Load the shine and flare textures into memory. */
  for (i = 0; i < 10; ++i)
    {
      snprintf (buffer, sizeof buffer,
		"flare-textures/Shine%d.jpg", i);

      if ((shine_tex[i] = xrTextureLoad (buffer, 0, 0, 0,
					 XR_TEXTURE_MIPMAPS|XR_TEXTURE_MODULATE)) < 0)
	return -1;
    }

  for (i = 0; i < 6; ++i)
    {
      snprintf (buffer, sizeof buffer,
		"flare-textures/Flare%d.jpg", i+1);

      if ((flare_tex[i] = xrTextureLoad (buffer, 0, 0, 0,
					 XR_TEXTURE_MIPMAPS|XR_TEXTURE_MODULATE)) < 0)
	return -1;
    }

  /* Initialize flare structures. */
  flare[0] = set_flare (-1, 1, 0.3, blue, 1.0);
  flare[1] = set_flare (-1, 1, 0.2, green, 1.0);
  flare[2] = set_flare (-1, 1, 0.25, red, 1.0);

  flare[3] = set_flare (2, 1.3, 0.04, red, 0.6);
  flare[4] = set_flare (3, 1, 0.1, red, 0.4);
  flare[5] = set_flare (1, 0.5, 0.2, red, 0.3);
  flare[6] = set_flare (3, 0.2, 0.05, red, 0.3);
  flare[7] = set_flare (0, 0, 0.04, red, 0.3);
  flare[8] = set_flare (5, -0.25, 0.07, red, 0.5);
  flare[9] = set_flare (5, -0.4, 0.02, red, 0.6);
  flare[10] = set_flare (5, -0.6, 0.04, red, 0.4);
  flare[11] = set_flare (5, -1, 0.03, red, 0.2);

  xrLogAssert (NUM_FLARES == 12);

  return 0;
}

void
flare_unload ()
{
  int i;

  /* Unload shine and flare textures. */
  for (i = 9; i >= 0; --i)
    xrTextureUnload (flare_tex[i]);
  for (i = 5; i >= 0; --i)
    xrTextureUnload (shine_tex[i]);
}

static void
do_flares (const GLfloat *eye, const GLfloat *centre,
	   const GLfloat *view_direction)
{
  GLfloat axis[3], position[3], dx[3], dy[3], sx[3], sy[3];
  int i;

  /* Normalize sun_direction to be axis. */
  xrNormalize (sun_direction, axis);

  /* Compute dx, dy to be parallel to viewport. */
  xrCrossProduct (view_direction, sun_direction, dx);
  xrCrossProduct (dx, view_direction, dy);

  glEnable (GL_BLEND);
  glBlendFunc (GL_ONE, GL_ONE);
  glEnable (GL_TEXTURE_2D);

  /* XXX Surely this should be saved as part of the texture state? */
  glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  for (i = 0; i < NUM_FLARES; ++i)
    {
      sx[0] = flare[i].scale * dx[0] * FLARE_SCALE_FACTOR;
      sx[1] = flare[i].scale * dx[1] * FLARE_SCALE_FACTOR;
      sx[2] = flare[i].scale * dx[2] * FLARE_SCALE_FACTOR;

      sy[0] = flare[i].scale * dy[0] * FLARE_SCALE_FACTOR;
      sy[1] = flare[i].scale * dy[1] * FLARE_SCALE_FACTOR;
      sy[2] = flare[i].scale * dy[2] * FLARE_SCALE_FACTOR;

      if (flare[i].type < 0)
	{
	  static int tick = 0;

	  glBindTexture (GL_TEXTURE_2D, shine_tex[tick]);
	  tick = (tick + 1) % 10;
	}
      else
	{
	  glBindTexture (GL_TEXTURE_2D, flare_tex[flare[i].type]);
	}

      /* Centre position of the flare. */
      position[0] = centre[0] + flare[i].loc * axis[0];
      position[1] = centre[1] + flare[i].loc * axis[1];
      position[2] = centre[2] + flare[i].loc * axis[2];

      /* Draw it. */
      glColor3fv (flare[i].colour);

      glBegin (GL_QUADS);
      glTexCoord2f (0, 0);
      glVertex3f (position[0] + sx[0] + sy[0],
		  position[1] + sx[1] + sy[1],
		  position[2] + sx[2] + sy[2]);
      glTexCoord2f (0, 1);
      glVertex3f (position[0] + sx[0] - sy[0],
		  position[1] + sx[1] - sy[1],
		  position[2] + sx[2] - sy[2]);
      glTexCoord2f (1, 1);
      glVertex3f (position[0] - sx[0] - sy[0],
		  position[1] - sx[1] - sy[1],
		  position[2] - sx[2] - sy[2]);
      glTexCoord2f (1, 0);
      glVertex3f (position[0] - sx[0] + sy[0],
		  position[1] - sx[1] + sy[1],
		  position[2] - sx[2] + sy[2]);
      glEnd ();
    }

  glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
  glDisable (GL_TEXTURE_2D);
  glDisable (GL_BLEND);
}

static void
do_alphas (GLfloat value)	/* value = range [0,1] */
{
  /* Switch to orthographic projection. */
  glMatrixMode (GL_PROJECTION);
  glPushMatrix ();
  glLoadIdentity ();
  glOrtho (0, (GLdouble) xrWidth, 0, (GLdouble) xrHeight, 0, 1000);
  glMatrixMode (GL_MODELVIEW);
  glPushMatrix ();
  glLoadIdentity ();

  /* Enable alpha blending. */
  glEnable (GL_BLEND);
  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  glBegin (GL_QUADS);
  glColor4f (1, 1, 1, 0.75 * value);
  glVertex2i (0, 0);
  glVertex2i (0, xrHeight);
  glVertex2i (xrWidth, xrHeight);
  glVertex2i (xrWidth, 0);
  glEnd ();

  glDisable (GL_BLEND);

  glMatrixMode (GL_MODELVIEW);
  glPopMatrix ();
  glMatrixMode (GL_PROJECTION);
  glPopMatrix ();
}

void
flare_display (const struct xrPlayer *player)
{
  GLfloat sun_intensity;
  GLfloat view_direction[3];
  const GLfloat *eye;
  const GLfloat *centre;

  /* Work out if we can see the sun from here. */
  eye = xrPlayerGetCameraEye (player, 0); /* XXX OOBE should be passed in. */
  centre = xrPlayerGetCameraCentre (player, 0);

  view_direction[0] = centre[0] - eye[0];
  view_direction[1] = centre[1] - eye[1];
  view_direction[2] = centre[2] - eye[2];

  xrNormalize (view_direction, view_direction);
  xrNormalize (sun_direction, sun_direction);
  sun_intensity = xrDotProduct (sun_direction, view_direction);

#if 0
  xrLog (LOG_DEBUG, "sun_intensity = %g", sun_intensity);
#endif

  /* If we can see the sun, display the flare. */
  if (sun_intensity > 0.80)	/* == approximately 30 degrees either side */
    {
      do_flares (eye, centre, view_direction);

      /* If we are pointing right at the sun, alpha the display.
       * (XXX This is not quite right - we should be able to arrange
       * to do this /after/ drawing all other components of the
       * view, else when there are other craft visible, this won't
       * look quite right).
       */
      if (sun_intensity > 0.93)
	{
	  do_alphas ((sun_intensity - 0.93) / 0.07);
	}
    }
}
