#ifndef GR_ACL_H
#define GR_ACL_H
#endif
#include <linux/grdefs.h>
#include <linux/resource.h>

#include <asm/resource.h>

/* * * * * * * * * * * * * * * * * * * * *
 * grsecurity ACL System
 * Main header file
 * Purpose: define most gracl data structures 
 * * * * * * * * * * * * * * * * * * * * */

/* Major status information */

#define GR_VERSION  "grsecurity 1.9.14"

enum {

	SHUTDOWN = 0,
	ENABLE = 1,
	GOD = 2,
	RELOAD = 3,
	SEGVMOD = 4
};

/* Password setup definitions
 * kernel/grhash.c */
enum {
	GR_PW_LEN = 128,
	GR_SALT_LEN = 16,
	GR_SHA_LEN = 32,
};

/* Begin Data Structures */

struct admin_pw {
	unsigned char salt[GR_SALT_LEN];
	unsigned char sum[GR_SHA_LEN];	/* 256-bit SHA hash of the password */
};

struct name_entry {
	ino_t inode;
	kdev_t device;
	char *name;
};

struct acl_subject_db {
	struct acl_subject_label **s_hash;
	__u32 s_size;
};

struct name_db {
	struct name_entry **n_hash;
	__u32 n_size;
};

struct crash_uid {
	uid_t uid;
	unsigned long expires;
};

/* Userspace Grsecurity ACL data structures */
struct acl_subject_label {
	char *filename;
	ino_t inode;
	kdev_t device;
	__u32 mode;
	__u32 cap_raise;
	__u32 cap_lower;

	struct rlimit res[RLIM_NLIMITS + 1];
	__u16 resmask;

	__u32 ip_proto[8];
	__u32 ip_type;
	struct acl_ip_label **ips;
	__u32 ip_num;

	__u32 crashes;
	unsigned long expires;

	struct acl_object_label *proc_object;
	struct acl_ip_label *ip_object;
	struct acl_subject_label *prev;
	struct acl_subject_label *next;

	struct acl_object_label **obj_hash;
	__u32 obj_hash_size;
};

struct user_acl_subject_db {
	struct acl_subject_label **s_table;
	__u32 s_entries;	/* Number of entries in table */
	__u32 i_entries;	/* total number of ip acls */
	__u32 o_entries;	/* Total number of object acls */
};

struct acl_object_label {
	char *filename;
	ino_t inode;
	kdev_t device;
	__u32 mode;

	/* next two structures not used */

	struct acl_object_label *prev;
	struct acl_object_label *next;
};

struct acl_ip_label {
	__u32 addr;
	__u32 netmask;
	__u16 low, high;
	__u8 mode;
	__u32 type;
	__u32 proto[8];

	/* next two structures not used */

	struct acl_ip_label *prev;
	struct acl_ip_label *next;
};

struct gr_arg {
	struct user_acl_subject_db subj_db;
	unsigned char pw[GR_PW_LEN];
	unsigned char salt[GR_SALT_LEN];
	unsigned char sum[GR_SHA_LEN];
	kdev_t segv_device;
	ino_t segv_inode;
	uid_t segv_uid;
	__u16 mode;

};

/* End Data Structures Section */

/* Hash functions generated by empirical testing by Brad Spengler
   Makes good use of the low bits of the inode.  Generally 0-1 times
   in loop for successful match.  0-3 for unsuccessful match.
   Shift/add algorithm with modulus of table size and an XOR*/

static __inline__ unsigned long
fhash(const ino_t ino, const kdev_t dev, const unsigned long sz)
{
	return (((ino + dev) ^ ((ino << 13) + (ino << 23) + (dev << 9))) % sz);
}

static __inline__ unsigned long
nhash(const char *name, const unsigned long sz)
{
	return full_name_hash(name, strlen(name)) % sz;
}
